@* Support Functions.
These are functions for convenience or easier handling of some
aspects of the triangulation. They are very short and easy to
implement, so they are not worth a chapter on each own and
gathered here together.


@ The following functions allow more comfort for input e.g. from mouse. It chooses
the point in the current hull which is closest to a given point x.

@<Further mem...@>+=
vector make_vector(const d_rat_point &p);

@ We use |vector|s to make life easier, but at the cost of some precision.
Although when you work with a mouse it is to hard to hit the same pixel twice
and for such cases this function is intended.

@<Member fun...@>+=
vector Triangulation::make_vector(const d_rat_point &p)
{
int i;
vector v(p.dim());

for (i=1;i<=p.dim();i++)
  v[i-1]=p[i].todouble()/p[0].todouble();
return v;
}

d_rat_point Triangulation::find_closest_point(const d_rat_point &x)
{
double dist;
d_rat_point p(dmax),cp(dmax);
vector d(dmax);
dist=MAXDOUBLE; // approximately $+\infty$

forall(p,coordinates)
 {
  d=make_vector(p-x);
  if (d.length()<dist)
   {
    dist=d.length();
    cp=p;
   }
 }
return cp;
}

@ Hyperplanes are only constructed when needed. These functions take care of it.

@<Further mem...@>+=
int sees_facet(Simplex *S,int f, const d_rat_point &x);
void compute_plane(Simplex *S,int j);

@ A hyperplane take all points of a simplex except one as its base. This special point
is the point describing it (i.e., has the same index). It lies in the positive
halfspace of the hyperplane. When this point is the |anti_origin|, we cannot
compute with it. Therefore we take the origin ($O =$ |quasi_center/(dcur+1)|)
 and place it on the negative
side of the hyperplane. When we are not in maximum dimension, the normal vector
has to lie in the affine hull of the current hull. This means, it is a
linear combination of the spanning vectors derived from the vertices of the
current simplex. When this is an unbounded simplex, we take the origin
instead of |anti_origin|. It is affinely independent of all other
vertices because it lies in the interior of the hull and the other
points on the edge.

When it is possible we take the shortcut of using the precomputed
plane of the neighbor simplex. It describes the opposite space but has
the same equation, so we simply copy it with the sign of the coefficients
inverted.

|sees_facet()| additionally returns the side of the hyperplane where |x|
lies. Often this is the only interest for computing a hyperplane.
\label{compute hyperplanes}

@<Member fun...@>+=
int Triangulation::sees_facet(Simplex *S,int f, const d_rat_point &x)
{
if (S->valid_equations[f] != dcur) 
  compute_plane(S,f);
return which_side(S->facets[f],x);
}

void Triangulation::compute_plane(Simplex *S, int f) @/
{
if (S->valid_equations[f] != dcur)
 {
  Simplex *NS;
  int o;

  NS=S->neighbors[f];
  o=S->opposite_vertices[f];
  if ((NS!=nil)&&(NS->valid_equations[o]==dcur)) // copy the reverse
   {
    S->facets[f]=NS->facets[o].reverse();
   }
  else  // compute ourselves
   {
    int i,j;
    array<d_rat_point> P(1,dcur);

    j=1;
    for (i=0; i<=dcur; i++)
      if (i!=f)
        P[j++] = coordinates[S->vertices[i]];
    if (dcur==dmax) {
      if (S->vertices[f]==anti_origin)
        S->facets[f]=hyperplane(P,quasi_center/(dcur+1),-1);
      else
        S->facets[f]=hyperplane(P,coordinates[S->vertices[f]],1);
    }
    else {
      array<integer_vector> N(1,dcur);
      d_rat_point p0;

      if (S->vertices[f]==anti_origin) @/
        p0=quasi_center/(dcur + 1); // works, we are convex!
      else
        p0=coordinates[S->vertices[f]];
      j=1;
      for(i=0;i<=dcur;i++)
        if (i!=f)
	  N[j++]=make_direction_from_point(coordinates[S->vertices[i]]-p0);
      if (S->vertices[f]==anti_origin)
        S->facets[f]=hyperplane(P,N,p0,-1);
      else
        S->facets[f]=hyperplane(P,N,p0,1);
    }
   }
  S->valid_equations[f]=dcur;
 }
}

@ We need a function |is_dimension_jump()|, which tells us
whether |x| is a dimension jump or not.
@<Further mem...@>+=
bool is_dimension_jump(const d_rat_point &x);

@
How can we test whether |x| is a dimension jump? |x| is a dimension
jump iff |x| does not lie in the affine hull of the vertices of the
\label{is_dimension_jump}origin simplex. Since all these vertices are affine-linearly
independent by our construction, we only have to test whether |x| and
all these vertices are affine-linearly dependent. We test this by
using the function |is_contained_in_affine_hull()|, which gets as argument an
array of all the vectors to test.
@<Member fu...@>+=
bool Triangulation::is_dimension_jump(const d_rat_point &x) @/
{@/
  array<d_rat_point> A(0,dcur);
  int i;

  for (i=0; i<=dcur; i++)
    A[i]=coordinates.contents(origin_simplex->vertices[i]);

  return !is_contained_in_affine_hull(A,x);
}


@ Sometimes it is useful to know whether a point is contained in the
convex hull. The tests to perform are similiar to those when inserting a point,
but no action is taken. The first case is an empty hull. It surely has no points in it.
The next case is that the point to test would be a dimension jump. If so it
can not be
a member of the hull. When we have come this far we know that |x| lies in the affine
hull of the triangulation. We call |find_visible_facets()| to see whether
it belongs to the interior of the hull. If not we have to |clear()| the list
and return false.

@<Member fun...@>+=

bool Triangulation::member(const d_rat_point &x)
{
if (dcur == -1)
  return false; // no points -- no members
if (dcur < dmax)
  if (is_dimension_jump(x))
    return false;
find_visible_facets(x);
if (visible_simplices.empty())
  return true;
visible_simplices.clear();
return false;
}


@ 
We sometimes want to sort lists of |list_item|s describing points according to their
|order_nr|. We use {\sc Leda}'s |bucket_sort()| for |list|s for this.
We have to declare the order function needed by bucketsort and the
|sort()| function using it. |curr_tria| is the replacement for the
|this|-pointer that we cannot use, because the |order_nof| function is called from 
within |bucket_sort()| and has to be static for this.

@<Further mem...@>+=
static int order_nof(const list_item &co);
static Triangulation *curr_tria;
void sort(list<list_item> &L);

@ This is the implementation. Static members are necessary because  they
are called from within |bucket_sort()| and cannot determine the object instance
(which triangulation) they belong to.

@<Member fun...@>+=
Triangulation *Triangulation::curr_tria; // we need to reserve space for it

int Triangulation::order_nof(const list_item &co)
{
return curr_tria->order_nr[co];
}

void Triangulation::sort(list<list_item> &L)
{
curr_tria=this;
L.bucket_sort(0,co_nr,order_nof);
}

@ This function tests whether two facets of two simplices are equal.

@<Further mem...@>+=
bool facets_equal(Simplex *sim1,int facet1,Simplex *sim2, int facet2);

@ Two facets are equal if they contain the same points. This test is an easy run through the
|vertices[]| array in each simplex omitting the vertex corresponding
to the facet being compared. This takes only $O(n^2)$ because we are only
comparing pointers (|list_item|) to vectors, not whole vectors themselves.
\label{facets equal}
Note also that we do not use the |operator==| for |hyperplane|s
because two facets may have the same affine hull
(|hyperplane|) but defined by different sets of points.

@<Member fun...@>+=

bool Triangulation::facets_equal(Simplex *sim1,int facet1,Simplex *sim2,int facet2)
{
int i,j;

for (i=0;i<=dcur;i++)
  if (i!=facet1)
   {
    for(j=0;j<=dcur;j++)
     {
      if (j==facet2)
        continue;
      if (sim1->vertices[i]==sim2->vertices[j])
        break;
     }
    if (j==dcur+1) // not found
      return false;
   }
return true; //all found
}

@ For test purposes we want to know all points inserted by now. So
here they are.

@<Member fun...@>+=
list<d_rat_point> Triangulation::points()
{
  return coordinates;
}

