


#ifndef LEDA_D_RAT_POINT_H
#define LEDA_D_RAT_POINT_H

#include <stream.h>
 
class d_rat_point;
class d_rat_segment;
class hyperplane;

int compare_help(const d_rat_point &p, const d_rat_point &q);
inline int compare(const d_rat_point &p, const d_rat_point &q)
{return compare_help(p,q);}


#include "numberdef"
#include <math.h>
#include <LEDA/integer.h>
#include <LEDA/rational.h>
#include <LEDA/array.h>
#include <LEDA/array2.h>
#include <LEDA/list.h>
#include "integer_vector.h"
#include <LEDA/handle_types.h>
#include <LEDA/plane.h>

enum kind{homogeneous,cartesian};


//------------------------------------------------------------------------------
// d_rat_points
//------------------------------------------------------------------------------

class d_rat_point_rep: public handle_rep
 {

friend class d_rat_point;
friend class d_rat_segment;
friend class hyperplane;

  
number* coordinates;
int dim;

/* dim is the dimension of the point and coordinates is the array
of homogeneous coordinates. We use coordinates[0] for the common
denominator of all cartesian coordinates; it is always non-zero.
the default constructor constructs a point with no cartesian coordinate
*/



public:

d_rat_point_rep() 
{ dim = 0;
  coordinates = new number[1];
  coordinates[0] = 1;
}



d_rat_point_rep(int d)  
   { dim = d;
     coordinates = new number[d+1];
     coordinates[0] = 1;
     for (int i = 1; i <= dim; i++) coordinates[i] = 0;
    }


d_rat_point_rep(int d, number D, const integer_vector& c) 
    { 
      int sig;

      dim = d;
 
      if (d != c.dim()) error_handler(1,"d_rat_point_rep:: wrong dimension of c in constructor");

      coordinates = new number[dim+1];

      sig = sign(D); 
      coordinates[0] = sig * D ;
      for (int i = 1; i <= dim; i++) coordinates[i] = sig * c[i - 1];
           
    }

d_rat_point_rep(int d, number D, const array<number>& L) 
    { dim = d;
      int low = L.low();
      int sig = sign(D);
 
      if (d != L.high() - low + 1) error_handler(1,"d_rat_point_rep:: wrong dimension of c in constructor");

      coordinates = new number[dim+1];

       
      coordinates[0] = sig * D;
      for (int i = 1; i <= dim; i++) coordinates[i] = sig * L[low - 1 + i];
           
    }

d_rat_point_rep(int d, const integer_vector& c) 
    { dim = d;
      coordinates = new number[dim+1];
      if (d == c.dim()) 
	  {coordinates[0] = 1;
           for (int i = 1; i <= dim; i++) coordinates[i] = c[i - 1];
           
          }
      else
          { if (c.dim() != dim + 1) error_handler(1,"d_rat_point_rep:: wrong dimension of c in constructor");

           int sig = sign(c[0]);
	   for (int i = 0; i <= dim; i++) coordinates[i] = sig * c[i];
          }
    }

d_rat_point_rep(int d, const array<number>& L) 
    { dim = d;
      coordinates = new number[dim+1];
      int low = L.low();
      int n = L.high() - L.low() +1;
      if (d == n) 
	  {coordinates[0] = 1;
           for (int i = 1; i <= dim; i++) coordinates[i] = L[low - 1 +i];
           
          }
      else
          { if (n != d +1) error_handler(1,"d_rat_point_rep:: wrong dimension of L in constructor");

	   int sig = sign(L[low]);
	   for (int i = 0; i <= dim; i++) coordinates[i] = sig * L[low +i];
          }
    }


d_rat_point_rep(const array<number>& L, kind k) 
{ 
      int low = L.low();
      int n = L.high() - L.low() +1;
      if (k == cartesian) 
	  {dim = n; coordinates = new number[dim+1];
           coordinates[0] = 1;
           for (int i = 1; i <= dim; i++) coordinates[i] = L[low - 1 +i];
           
          }
      else
          {  dim = n -1 ;
             coordinates = new number[dim+1];
             int sig = sign(L[low]);
              for (int i = 0; i <= dim; i++) coordinates[i] = sig * L[low +i];
          }
    }



~d_rat_point_rep(){ delete[] coordinates;}
	  




LEDA_MEMORY(d_rat_point_rep)

friend d_rat_point operator/(const d_rat_point & p, number n);
 


};



/*{\Manpage {rat\_point} {} {Arbitrary-Dimensional Rational Points} }*/

class d_rat_point: public handle_base
 {

/*{\Mdefinition
An instance of the data type $d\_rat\_point$ is a point with rational coordinates 
in an arbitrary dimensional space. A point $(x_1,\ldots,x_d)$ is represented by homogeneous
coordinates $(h_0,h_1,\ldots,h_d)$ of arbitrary length integers (see \ref{Integers of 
Arbitrary Length}) such that $x_i = h_i/h_0$. }*/


friend class d_rat_segment;
friend class hyperplane;

d_rat_point_rep* ptr() const { return (d_rat_point_rep*)PTR;} 

public:



/*{\Mcreation p}*/

d_rat_point() { PTR = new d_rat_point_rep; }
/*{\Mcreate introduces a variable \var\ of type \name\
initialized to a 0-dimensional point.}*/



d_rat_point(int d)       { PTR = new d_rat_point_rep(d); }
/*{\Mcreate introduces a variable \var\ of type \name\
initialized to the origin in d-dimensional space.}*/

d_rat_point(int d,number D, const integer_vector & c)
{ PTR = new d_rat_point_rep(d, D, c); }
/*{\Mcreate introduces a variable \var\ of type \name\
initialized to the point with homogeneous coordinates
$(D,c[0],\ldots,c[d-1])$.\\
\precond $d$ is the dimension of $c$}*/

d_rat_point(int d,number D, const array<number> & L)
{ PTR = new d_rat_point_rep(d, D, L); }
/*{\Mcreate introduces a variable \var\ of type \name\
initialized to the point with homogeneous coordinates
$(D,L[low],\ldots,L[high])$  wher $low$ and $high$ are the index bounds
$L$.
\precond $ d = high - low + 1$.}*/



d_rat_point(int d, const integer_vector & c)
{ PTR = new d_rat_point_rep(d,c); }
/*{\Mcreate introduces a variable \var\ of type \name\
initialized to the point with cartesian coordinates
$c[0]$ to $c[c.dim()-1]$ if $c.dim() = d$
and to the point with homogeneous coordinates
$c[0]$ to $c[c.dim()-1]$ if $c.dim() = d + 1$.}*/

d_rat_point(int d, const array<number> & L)
{ PTR = new d_rat_point_rep(d,L); }
/*{\Mcreate introduces a variable \var\ of type \name\
initialized to the point with cartesian coordinates
$L[low]$ to $L[high]$ if $n = d$
and to the point with homogeneous coordinates
$L[low]$ to $L[high]$ if $n = d + 1$. Here $n = high - low +1$ and
$low$ and $high$ are the index bounds of $L$.}*/

d_rat_point(const array<number> & L, kind k)
{ PTR = new d_rat_point_rep(L,k); }
/*{\Mcreate introduces a variable \var\ of type \name\
initialized to the point with cartesian coordinates
$L[low]$ to $L[high]$ if $k$ is "cartesian"
and to the point with homogeneous coordinates
$L[low]$ to $L[high]$ if $k$ is "homogeneous";
$low$ and $high$ are the index bounds of $L$.}*/



d_rat_point(const d_rat_point& p) : handle_base(p) {}
~d_rat_point()        {}        //  { clear(); }  still needed?

d_rat_point& operator=(const d_rat_point& p) 
{ handle_base::operator=(p); return *this; }


/*{\Moperations 2 3.5 }*/

rational ccoord(int i) const { return ptr()->coordinates[i]/ ptr()->coordinates[0]; }
/*{\Mop     returns the $i$-th cartesian coordinate of \var.}*/

number hcoord(int i) const { return ptr()->coordinates[i]; }
/*{\Mop     returns the $i$-th homogeneous coordinate of \var.}*/

number operator[](int i)const {return ptr()->coordinates[i];}
/*{\Marrop     returns the $i$-th homogeneous coordinate of \var.}*/

int dim() const {return ptr()->dim;}
/*{\Mop     returns the dimension of \var.}*/


friend d_rat_point operator/(const d_rat_point & p, number n);
/*{\Mbinopfunc     divides all cartesian coordinates by $n$.}*/


friend d_rat_point operator+(const d_rat_point & p, const d_rat_point & q);
/*{\Mbinopfunc  adds the cartesian coordinates   }*/

d_rat_point& operator+=(const d_rat_point& p); 

friend d_rat_point operator-(const d_rat_point & p, const d_rat_point & q);
/*{\Mbinopfunc  subtracts the cartesian coordinates   }*/

d_rat_point& operator-=(const d_rat_point& p); 

friend integer_vector minus(const d_rat_point & p, const d_rat_point & q);
/*{\Mbinopfunc  returns the direction $p -q$.   }*/


friend bool identical(const d_rat_point& p, const d_rat_point& q)
{ return p.ptr() == q.ptr(); }
/*{\Mfuncl  test for identity}*/

friend bool operator==(const d_rat_point& p, const d_rat_point& q);
/*{\Mbinopfunc  test for equality.}*/

friend bool operator!=(const d_rat_point& p, const d_rat_point& q);
/*{\Mbinopfunc  test for inequality.}*/

friend ostream& operator<<(ostream& O, const d_rat_point& p) ;
/*{\Mbinopfunc  writes the homogenous coordinates  of \var\ to 
                output stream $O$.}*/

friend istream& operator>>(istream& I, d_rat_point& p) ;
/*{\Mbinopfunc  reads the cartesian coordinates of \var\ from 
                input stream $I$. This operator uses the current dimension
                of  \var.}*/



//friend int orientation(const array<d_rat_point> & A);

friend integer_vector make_direction_from_point(const d_rat_point & p);
/*{\Mfuncl returns the integer vector with entries $p[1]$ to
$p[d]$ }*/


//friend bool is_contained_in_affine_hull(const array<d_rat_point> & A,d_rat_point x);

//friend bool is_contained_in_linear_hull(const array< d_rat_point>& A,d_rat_point x);

//friend int compare_help(const d_rat_point&, const d_rat_point&);

int cmp(const d_rat_point&, const d_rat_point&);  //hier stand eine static davor

};

LEDA_HANDLE_TYPE(d_rat_point)

inline void Print(const d_rat_point& p, ostream& out) { out << p; } 
inline void Read(d_rat_point& p,  istream& in)        { in >> p; }


int orientation(const array<d_rat_point> & A);
/*{\Mfuncl determines the orientation of the points in $A$, where $A$
consists of $d+1$ $d$-dimensional points }*/

int incircle(const array<d_rat_point> & A,const d_rat_point &x);
/*{\Mfuncl determines whether the point $x$ lies inside (= -1), on (= 0),
or outside (= +1) the circle defined by the points in $A$, where $A$
consists of $d+1$ $d$-dimensional points }*/


bool is_contained_in_linear_hull(const array< d_rat_point>& A,d_rat_point x);
/*{\Mfuncl determines whether $x$ is contained in the linear hull
of the points in $A$ }*/

bool is_contained_in_affine_hull(const array< d_rat_point>& A,d_rat_point x);
/*{\Mfuncl determines whether $x$ is contained in the affine hull
of the points in $A$ }*/

bool is_contained_in_simplex(const array< d_rat_point>& A,d_rat_point x);
/*{\Mfuncl determines whether $x$ is contained in the simplex spanned
by the points in |A|. |A| may consists of up to $d + 1$ points.
\precon The points in |A| are affinely independent.}*/



bool linearly_independent(const array<d_rat_point>& A);
/*{\Mfuncl decides whether the points in $S$ are linearly indenpendent}*/

bool affinely_independent(const array<d_rat_point>& A);
/*{\Mfuncl decides whether the points in $S$ are affinely indenpendent}*/

int affine_rank(const array<d_rat_point>& A);
/*{\Mfuncl computes the affine rank of the points in $A$}*/

int  linear_rank(const array<d_rat_point>& A);
/*{\Mfuncl computes the linear rank of the points in $A$}*/

list<d_rat_point> linear_base(const array<d_rat_point>& A);
/*{\Mfuncl computes a basis of the linear space spanned by the points in $A$}*/



//inline int compare_help(const d_rat_point& p, const d_rat_point& q)
//{ return (p.ptr() == q.ptr()) ? 0 : d_rat_point::cmp(p,q); }



inline bool operator==(const d_rat_point& p, const d_rat_point& q)
{ return compare(p,q) == 0; }

inline bool operator!=(const d_rat_point& p, const d_rat_point& q)
{ return compare(p,q) != 0; }


inline point convert(d_rat_point x)
{
#ifdef DOUBLE
return point(x[1]/x[0],x[2]/x[0]);
#endif

#ifdef INTEGER
return point(x[1].todouble()/x[0].todouble(),x[2].todouble()/x[0].todouble());
#endif
}

inline int Hash(const d_rat_point &x)
{
printf("mz Hash\n");
if (x.dim() == 0)
  return x[0].tolong();
else
  return x[1].tolong();
}




#endif





