\documentstyle[comments]{cweb}

@i /KM/usr/mehlhorn/Programs/typnamen

\begin{document}


@ I define hyperplane.c

@c
#include "numberdef"
#include "hyperplane.h"
#include "integer_matrix.h"
#include <LEDA/integer.h>

@<functions@>



@ Let us fill in all the functions that were not defined
in the header file. 

We start with the constructor that takes a set |P| of points 
and a set |N| of column vectors, a point |o|
and a sign |s| in $\{-1,+1\}$ and constructs a hyperplane whose
normal is a linear combination of the vectors in |N|, which passes through the
points in |P|, and which has |o| on the $s$-side. Let |n| be the number
of vectors and let |p| be the number of points. We assume |n| and |p| to
be at least one. We set up a linear system in $p + n$ unknowns expressing the
fact that we want a linear combination of the vectors in $N$ that 
satisfies $p$ plane equations $\sum p_{i,j} x_i = 0$.
This system has $p + n$ equations. We solve this 
system and then extract the coefficients from the solution.
We then normalize |normal| such that |o| lies on the correct side.
We set up and solve a system 
$$M \cdot \left( \begin{array}{c} 
c_0 \\ \vdots \\ c_{dim} \\
\lambda_1 \\ \vdots \\ \lambda_n
\end{array} \right) = 0 $$
where
$$ M = \left( \begin{array}{*{7}{c}}
p_{1,0} & \multicolumn{2}{c}{\ldots} & p_{1,dim} & 0 & \ldots & 0 \\
\vdots  & \multicolumn{2}{c}{}       & \vdots    & \vdots & & \vdots \\
p_{p,0} & \multicolumn{2}{c}{\ldots} & p_{p,dim} & 0 & \ldots & 0 \\
0      & 1 &        & 0 & n_{1,1}   & \ldots & n_{n,1} \\
\vdots &   & \ddots &   & \vdots    &        & \vdots \\
0      & 0 &        & 1 & n_{1,dim} & \ldots & n_{n,dim}
\end{array} \right) $$
We extract $c_0, \ldots, c_{dim}$ as our solution and disgard the
$\lambda$'s.



@<functions@>=

hyperplane_rep::hyperplane_rep(const array<d_rat_point>&  P, 
const array<integer_vector> &  N, const d_rat_point & o, int side)
{
int d = o.dim(); // we are in $d$-dimensional space
int p = P.high() - P.low() + 1;
int plow = P.low();
int n = N.high() - N.low() + 1;
int nlow = N.low();
int i,j;

dim = d;
coefficients = new number[dim + 1];
for (i=0;i<p;i++)
  if (P[plow+i].dim()!=d)
    error_handler(1,"hyperplane::constructor: P-dim does not fit");
for (i=0;i<n;i++)
  if (N[nlow+i].dim()!=d)
    error_handler(1,"hyperplane::constructor: N-dim does not fit");

integer_matrix A(int(p+d), int(d+1+n));
 /* rows,colummns */

// upper part
// copy points to M
for(i=0;i<p;i++)
  for (j=0;j<=d;j++)
    A(i,j)=P[i+plow][j];

// lower part
for(i=0;i<d;i++)
 {
  A(p+i,1+i)=1;
  for(j=0;j<n;j++)
    A(p+i,d+1+j)=N[nlow+j][i];
 }

integer_vector non_triv_sol;
homogeneous_linear_solver(A, non_triv_sol);
number sum = 0;

for (i=0;i<=d;i++)
  sum+=non_triv_sol[i] * o[i];
if (sum==0)
  error_handler(1,"cannot use o to determine side");
int o_side = sign(sum) * sign (o[0]);
if (side != o_side)
  for(i=0;i<=d;i++)
    coefficients[i]=-non_triv_sol[i];
else
  for(i=0;i<=d;i++)
    coefficients[i]=non_triv_sol[i];
}


@ Next comes the constructor that takes a set of $d$ points in 
$d$-dimensional space. I simply have to find a 
vector $n$ such that $p^T \cdot n = 0$ for every point in $P$. Afterwards I use
the point |o| to normalize. 


@<functions@>+=

hyperplane_rep::hyperplane_rep(const array<d_rat_point>&  P, 
const d_rat_point & o, int side)

{ 
  int d = dim = o.dim();
  if (d != P.high() - P.low() +1) 
    error_handler(1,"hyperplane-constructor: P has wrong number of points");
  coefficients = new number[d+1];

  int plow = P.low();
  integer_matrix A(d,d+1);
  
  int i,j,k;
  for (i = 0; i < d; i++)
  { /* define $i$-th equation */
   for (j = 0; j <=d; j++) A(i,j) = P[plow  + i][j]; //$j$-th coord of $i$-th point
  }

integer_vector non_triv_sol;
homogeneous_linear_solver(A,non_triv_sol);

number sum = 0;
for (i = 0; i<=d;i++) sum += non_triv_sol[i]*o[i];
if (sum == 0) error_handler(1,"cannot use o to determine side");

int o_side = (sum > 0? 1 : -1) * (o[0]>0? 1 : -1);
if (side != o_side)
 {for(i = 0; i <= d; i++) coefficients[i] = - non_triv_sol[i];}
else {for(i = 0; i <= d; i++) coefficients[i] = non_triv_sol[i];}


}
@ The enxt two functions return the equation and the normal vector of a
hyperplane respectively. The first function simply turns |coefficients|
into a LEDA-vector. For the second function we observe that any vector 
$\lambda (c1,\ldots cd)$ is a normal vector. We want to point the vector
from the negative to the positive halfspace. Recall that our hyperplane
has the equation $c[0] + sum (c[i]*z[i])  = 0$, where the $z[i]$ are Euclidian
point coordinates. The point $z = -c0\cdot c/{\parallel c \parallel}$ is
on the hyperplane, the point $z_n = (-1 -c0)\cdot c/{\parallel c \parallel}$
is in the negative halfspace and the point 
$z_p = (1 - c0)\cdot c/{\parallel c \parallel}$ is in the positive halfspace.
Thus the desired vector is simply $c$.

The function |value_at| returns $\sum_{0 \le i \le d} h_ip_i$.

@<functions@>+=

integer_vector hyperplane::equation() const
{ integer_vector result(ptr()->dim + 1);
for (int i = 0; i <= ptr()->dim; i++) {result[i] = ptr()->coefficients[i];}
return result;
}

integer_vector hyperplane::normal() const
{ integer_vector result(ptr()->dim);
for (int i = 0; i < ptr()->dim; i++) {result[i] = ptr()->coefficients[i+ 1];}
return result;

}

number hyperplane::value_at(d_rat_point p) const
{if (dim() != p.dim()) error_handler(1,"hyperplane::value_at: dimensions do not agree");
number result = 0;
for (int i = 0; i <= dim(); i++)
   {result +=  ptr()->coefficients[i] * p[i];}
return result;
}

@ The which side function is easy. I simply plug the point into the hyperplane
equation and test the sign of the cross product. Of course, i do not forget
that this amounts to a multiplication by the zeroeth coordinate of the point.


@<functions@>+=
int which_side( const hyperplane & h, const d_rat_point & x)
{ number Sum = 0;
if (h.dim() != x.dim()) error_handler(1,"hyperplane:which side incompatible dimension");
for(int i = 0; i <= h.dim(); i++) Sum += h[i]*x[i];
if (Sum == 0) return 0;
return ((Sum > 0? 1 : -1)*(x[0]>0? 1 : -1));
}

ostream& operator<<(ostream& out ,const hyperplane& h)
{ out << "(" ;
  for(int i = 0; i <= h.dim(); i++) cout << h[i] << (i < h.dim()? ",":"");
  out << ")";
  return out;
 } 


istream& operator>>(istream& in,hyperplane& h)
{cout << "not implemented yet";
return in;
}

int compare_help(const hyperplane &p, const hyperplane &q)
{error_handler(1,"not implemented"); return 1;}


@
\end{document}
